/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   CAN King Protocol API Definition
::   Copyright   :   (C)2003-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file CANKing.h
    \brief The CAN King protocol is a peer-to-peer protocol used to configure and exercise a CAN Kingdom
           
           The protocol provides hooks to allow an external connection to request the application to provide
           certain operations like device control or data requests. The protocol itself only understands that
           there has been a request but does not understand the context of that request. Consequently it must
           makes calls to the application through supplied event callback functions. The application configures
           these callbacks when it first registers the archive or when it registers the protocol.
           
           See RegisterMotoTronProtocolArchive() and the \ref S_RegisterProtocolArchiveInfo data structure. */

#ifndef __CANKING_H
#define __CANKING_H

/*---- INCLUDE FILES --------------------------------------------------------------------------------------*/
#include <typedefn.h>

/*---- DEFINES --------------------------------------------------------------------------------------------*/

/*---- TYPEDEF --------------------------------------------------------------------------------------------*/
#pragma pack(1)

typedef enum
{
    CONTROLREQUEST_RELINQUISH,
    CONTROLREQUEST_REQUEST,
    CONTROLREQUEST_FUNCTION_DATA
} E_ControlRequests;

/*! Describes the data structure associated with a \ref CANKING_PROTOCOL data request */
typedef struct
{
    uint1   u1Function;         /*!< The function */
    uint1   u1ControlRequest;   /*!< The control */
    uint1*  pu1Response;        /*!< The applications response to the data request. This is filled in by the
                                     application when the data request is attended to */
} S_ControlRequest;

/*! Describes the data structure associated with a \ref CANKING_PROTOCOL data request */
typedef struct
{
    uint1 u1RequestedDocument;  /*!< The requested document */
    uint1 u1RequestedData;      /*!< The requested data */
    uint1 u1RequestedItem;      /*!< The requested item */
    uint1 u1Response;           /*!< The applications response to the data request. This is filled in by the
                                     application when the data request is attended to */
} S_DataRequest;

/*! \brief Prototype of the callback function that is executed when the authority level of a
    \ref CANKING_PROTOCOL connection is successfully increased.
    
    The city that increased its authority along with the old and new authority is provided as parameters of
    the callback function. */
    typedef void (*CANKingAuthorityIncreasedCbackFuncPtr)(NativeVar_U in_ConnectedCity, NativeVar_U in_uOldAthorityLevel, NativeVar_U in_uNewAuthorityLevel);

/*! \brief Prototype of the callback function that will be executed each time the authority level of a
    \ref CANKING_PROTOCOL connection is successfully negotiated.
    
    The city that established the connection and the current authority level of that connection are supplied
    as parameters when the callback executes. */
    typedef void (*CANKingConnectCbackFuncPtr)(NativeVar_U in_ConnectedCity, NativeVar_U in_uAuthorityOfConnection);

/*! \brief Prototype of the callback function that will be executed when a \ref CANKING_PROTOCOL data request is made.

    The city that made the request and a pointer to a buffer that contains the request information are supplied
    as parameters of the callback function. At a minimum the application should supply a value for the
    S_DataRequest::u1Response member pointed to by \p inout_pDataRequestObj */
    typedef void (*CANKingDataRequestCbackFuncPtr)(NativeVar_U in_uRequestingCity, S_DataRequest* inout_pDataRequestObj);

/*! \brief Prototype of the callback function that will be executed when a \ref CANKING_PROTOCOL control request is made.

    The city that made the request and a pointer to a buffer that contains the request information are supplied
    as parameters of the callback function. At a minimum the application should supply a value for the
    S_DataRequest::pu1Response member pointed to by \p inout_pDataRequestObj */
    typedef void (*CANKingControlRequestCbackFuncPtr)(NativeVar_U in_uRequestingCity, S_ControlRequest* inout_pControlRequestObj);

/*! \brief Prototype of the callback function that will be executed when a \ref CANKING_PROTOCOL change of
           authority request is made by a connected city.

    The requested authority is supplied along with a pointer to a buffer that holds the seed and returns with the result.*/
    typedef NativeBool (*CANKingAuthoritySecurityCbackFuncPtr)(uint1 in_u1Authority, uint4* inout_pu4SeedAndResult);

#pragma pack()
/*---- EXTERNALS ------------------------------------------------------------------------------------------*/

/*---- PROTOTYPES -----------------------------------------------------------------------------------------*/

#endif /* __CANKINGPROTOCOL_H */

/*---- END OF FILE ----------------------------------------------------------------------------------------*/

